using System;
using System.Collections.Generic;
using System.Globalization;

/// <summary>
/// Ponco - Approximate 0/1 Knapsack
/// Multiple strategies combined for best result
/// </summary>
class Ponco
{
    static int n;
    static long W;
    static double epsilon;
    static long[] weight, value;
    
    // Meet-in-the-Middle for exact solution (n <= 45)
    static (long val, List<int> items) MeetInMiddle()
    {
        int half = n / 2;
        
        var left = new List<(long w, long v, long mask)>();
        for (long mask = 0; mask < (1L << half); mask++)
        {
            long w = 0, v = 0;
            for (int i = 0; i < half; i++)
            {
                if ((mask & (1L << i)) != 0)
                {
                    w += weight[i];
                    v += value[i];
                }
            }
            if (w <= W) left.Add((w, v, mask));
        }
        
        left.Sort((a, b) => a.w.CompareTo(b.w));
        
        // Build monotonic hull
        var hull = new List<(long w, long v, long mask)>();
        long bestV = -1;
        foreach (var x in left)
        {
            if (x.v > bestV)
            {
                bestV = x.v;
                hull.Add(x);
            }
        }
        
        int right = n - half;
        long maxVal = 0;
        long leftMask = 0, rightMask = 0;
        
        for (long mask = 0; mask < (1L << right); mask++)
        {
            long w = 0, v = 0;
            for (int i = 0; i < right; i++)
            {
                if ((mask & (1L << i)) != 0)
                {
                    w += weight[half + i];
                    v += value[half + i];
                }
            }
            if (w > W) continue;
            
            long rem = W - w;
            int lo = 0, hi = hull.Count - 1, best = -1;
            while (lo <= hi)
            {
                int mid = (lo + hi) / 2;
                if (hull[mid].w <= rem) { best = mid; lo = mid + 1; }
                else hi = mid - 1;
            }
            
            long total = v + (best >= 0 ? hull[best].v : 0);
            if (total > maxVal)
            {
                maxVal = total;
                leftMask = best >= 0 ? hull[best].mask : 0;
                rightMask = mask;
            }
        }
        
        var items = new List<int>();
        for (int i = 0; i < half; i++)
            if ((leftMask & (1L << i)) != 0) items.Add(i + 1);
        for (int i = 0; i < right; i++)
            if ((rightMask & (1L << i)) != 0) items.Add(half + i + 1);
        
        return (maxVal, items);
    }
    
    // FPTAS with careful scaling
    static (long val, List<int> items) FPTAS()
    {
        long maxV = 0;
        for (int i = 0; i < n; i++)
            if (value[i] > maxV) maxV = value[i];
        
        if (maxV == 0) return (0, new List<int>());
        
        // K = eps * maxV / n, scaled = v / K = v * n / (eps * maxV)
        // We want scaled values in range [1, n/eps]
        
        int[] scaled = new int[n];
        long totalScaled = 0;
        
        for (int i = 0; i < n; i++)
        {
            double sv = (double)value[i] * n / (epsilon * maxV);
            scaled[i] = (int)Math.Max(1, Math.Min(sv, n / epsilon));
            totalScaled += scaled[i];
        }
        
        int dpSize = (int)Math.Min(totalScaled + 1, 10000001L);
        
        long[] dp = new long[dpSize];
        int[] lastItem = new int[dpSize];
        int[] prevVal = new int[dpSize];
        
        for (int i = 0; i < dpSize; i++)
        {
            dp[i] = long.MaxValue;
            lastItem[i] = -1;
            prevVal[i] = -1;
        }
        dp[0] = 0;
        
        int maxReached = 0;
        
        for (int i = 0; i < n; i++)
        {
            int sv = scaled[i];
            long wi = weight[i];
            
            int upperLimit = Math.Min(maxReached + sv, dpSize - 1);
            for (int v = upperLimit; v >= sv; v--)
            {
                int pv = v - sv;
                if (dp[pv] != long.MaxValue)
                {
                    long newW = dp[pv] + wi;
                    if (newW <= W && newW < dp[v])
                    {
                        dp[v] = newW;
                        lastItem[v] = i;
                        prevVal[v] = pv;
                        if (v > maxReached) maxReached = v;
                    }
                }
            }
        }
        
        int bestV = 0;
        for (int v = maxReached; v >= 0; v--)
        {
            if (dp[v] != long.MaxValue) { bestV = v; break; }
        }
        
        var items = new List<int>();
        int cur = bestV;
        while (cur > 0 && lastItem[cur] >= 0)
        {
            items.Add(lastItem[cur] + 1);
            cur = prevVal[cur];
        }
        
        long realVal = 0;
        foreach (int idx in items) realVal += value[idx - 1];
        
        return (realVal, items);
    }
    
    // Greedy by ratio
    static (long val, List<int> items) Greedy()
    {
        var order = new int[n];
        var ratios = new double[n];
        for (int i = 0; i < n; i++)
        {
            order[i] = i;
            ratios[i] = (double)value[i] / weight[i];
        }
        Array.Sort(ratios, order);
        Array.Reverse(order);
        
        long totalW = 0, totalV = 0;
        var items = new List<int>();
        
        foreach (int i in order)
        {
            if (totalW + weight[i] <= W)
            {
                totalW += weight[i];
                totalV += value[i];
                items.Add(i + 1);
            }
        }
        
        return (totalV, items);
    }
    
    // Greedy by value
    static (long val, List<int> items) GreedyValue()
    {
        var order = new int[n];
        var vals = new long[n];
        for (int i = 0; i < n; i++)
        {
            order[i] = i;
            vals[i] = value[i];
        }
        Array.Sort(vals, order);
        Array.Reverse(order);
        
        long totalW = 0, totalV = 0;
        var items = new List<int>();
        
        foreach (int i in order)
        {
            if (totalW + weight[i] <= W)
            {
                totalW += weight[i];
                totalV += value[i];
                items.Add(i + 1);
            }
        }
        
        return (totalV, items);
    }
    
    // Greedy by inverse weight (lightest first)
    static (long val, List<int> items) GreedyLight()
    {
        var order = new int[n];
        var wts = new long[n];
        for (int i = 0; i < n; i++)
        {
            order[i] = i;
            wts[i] = weight[i];
        }
        Array.Sort(wts, order);
        
        long totalW = 0, totalV = 0;
        var items = new List<int>();
        
        foreach (int i in order)
        {
            if (totalW + weight[i] <= W)
            {
                totalW += weight[i];
                totalV += value[i];
                items.Add(i + 1);
            }
        }
        
        return (totalV, items);
    }
    
    // Single best
    static (long val, List<int> items) SingleBest()
    {
        long bestV = 0;
        int bestI = -1;
        for (int i = 0; i < n; i++)
        {
            if (weight[i] <= W && value[i] > bestV)
            {
                bestV = value[i];
                bestI = i;
            }
        }
        if (bestI < 0) return (0, new List<int>());
        return (bestV, new List<int> { bestI + 1 });
    }
    
    // Take all items that fit
    static (long val, List<int> items) TakeAll()
    {
        long totalW = 0, totalV = 0;
        var items = new List<int>();
        
        for (int i = 0; i < n; i++)
        {
            if (totalW + weight[i] <= W)
            {
                totalW += weight[i];
                totalV += value[i];
                items.Add(i + 1);
            }
        }
        
        return (totalV, items);
    }
    
    static void Main()
    {
        var first = Console.ReadLine().Split(new[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
        n = int.Parse(first[0]);
        W = long.Parse(first[1]);
        epsilon = double.Parse(first[2], CultureInfo.InvariantCulture);
        
        weight = new long[n];
        value = new long[n];
        
        for (int i = 0; i < n; i++)
        {
            var parts = Console.ReadLine().Split(new[] { ' ', '\t' }, StringSplitOptions.RemoveEmptyEntries);
            weight[i] = long.Parse(parts[0]);
            value[i] = long.Parse(parts[1]);
        }
        
        // Collect all strategies
        var results = new List<(long val, List<int> items)>();
        
        if (n <= 45)
            results.Add(MeetInMiddle());
        
        results.Add(FPTAS());
        results.Add(Greedy());
        results.Add(GreedyValue());
        results.Add(GreedyLight());
        results.Add(SingleBest());
        results.Add(TakeAll());
        
        // Pick best valid result
        (long val, List<int> items) best = (0, new List<int>());
        
        foreach (var r in results)
        {
            // Validate
            long checkW = 0;
            bool valid = true;
            var seen = new HashSet<int>();
            
            foreach (int idx in r.items)
            {
                if (idx < 1 || idx > n || seen.Contains(idx))
                {
                    valid = false;
                    break;
                }
                seen.Add(idx);
                checkW += weight[idx - 1];
                if (checkW > W)
                {
                    valid = false;
                    break;
                }
            }
            
            if (valid && r.val > best.val)
                best = r;
        }
        
        Console.WriteLine(best.items.Count);
        Console.WriteLine(best.items.Count > 0 ? string.Join(" ", best.items) : "");
    }
}